<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Spatie\Sitemap\SitemapGenerator;

class SettingsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index($setting_name = 'web-information')
    {
        return view('admin.settings.index', [
            'backups' => $this->getBackups(),
            'route' => $setting_name,
        ]);
    }

    public function theme()
    {
        return view('admin.settings.theme');
    }

    /**
     *Change the Application Name
     */
    public function appInfoUpdate(Request $request)
    {
        $request->validate([
            'app_name' => 'required|min:4|string',
            'company_name' => 'required|min:4|string',
            'app_url' => 'required|string',
            'app_description' => 'nullable|string',
            'app_keyword' => 'nullable|string',
            'credit_footer' => 'required|string',
        ]);

        $this->updateSettings($request->except(['_token']));

        return redirect()->back()->with('success', 'App Information changed successfully');
    }

    /**
     *Change the Application Name
     */
    public function appMail(Request $request)
    {
        $request->validate([
            'MAIL_HOST' => 'required|string',
            'MAIL_PORT' => 'required|numeric',
            'MAIL_ENCRYPTION' => 'required|string',
            'MAIL_USERNAME' => 'required|string',
            'MAIL_PASSWORD' => 'required|string',
            'MAIL_FROM_ADDRESS' => 'required|string',
            'MAIL_FROM_NAME' => 'required|string',
        ]);
        foreach ($request->types as $key => $type) {
            $this->overWriteEnvFile($type, $request[$type]);
        }

        $this->updateSettings([
            'mail_host' => $request->MAIL_HOST,
            'mail_port' => $request->MAIL_PORT,
            'mail_encryption' => $request->MAIL_ENCRYPTION,
            'mail_username' => $request->MAIL_USERNAME,
            'mail_password' => $request->MAIL_PASSWORD,
            'mail_form_address' => $request->MAIL_FROM_ADDRESS,
            'mail_from_name' => $request->MAIL_FROM_NAME,
        ]);

        return redirect()->back()->with('success', 'Web Mail Information Updated successfully');
    }

    public function appContactUpdate(Request $request)
    {
        $data = $request->validate([
            'street' => 'required|string',
            'sitephone' => 'required|string',
            'country' => 'required|string',
            'state' => 'required|string',
            'city' => 'required|string',
            'postal_code' => 'required|string',
            'fulladdress' => 'nullable|string',
            'contactdescription' => 'nullable|string',
            'siteemail' => 'required|string',
            'facebook' => 'nullable|string',
            'twitter' => 'nullable|string',
            'youtube' => 'nullable|string',
            'instagram' => 'nullable|string',
            'linkedin' => 'nullable|string',
            'telegram' => 'nullable|string',
            'whatsapp' => 'nullable|string',
        ]);

        $this->updateSettings($data);

        return redirect()->back()->with('success', 'App Information changed successfully');
    }

    /**
     *Change the Application Logo
     */
    public function appPropertiesUpdate(Request $request)
    {
        $files = $request->validate([
            'app_dark_logo' => 'nullable|max:2048|image|mimes:jpeg,png,jpg',
            'app_light_logo' => 'nullable|max:2048|image|mimes:jpeg,png,jpg',
            'footer_dark_logo' => 'nullable|max:2048|image|mimes:jpeg,png,jpg',
            'footer_light_logo' => 'nullable|max:2048|image|mimes:jpeg,png,jpg',
            'favicon' => 'nullable|max:2048|image|mimes:jpeg,png,jpg,ico,bmp',
            'ogimage' => 'nullable|max:2048|image|mimes:jpeg,png,jpg',
            'short_logo' => 'nullable|max:2048|image|mimes:jpeg,png,jpg',
            'avatar' => 'nullable|max:2048|image|mimes:jpeg,png,jpg',
        ]);
        $logoPath = 'uploads/appLogo/';
        foreach ($files as $i => $file) {
            $name = uniqueId().'.'.$file->getClientOriginalExtension();
            $file->move($logoPath, $name);
            $oldName = explode('/', setting($i));

            $path = public_path($logoPath.end($oldName));
            if (file_exists($path) && is_file($path)) {
                unlink($path);
            }
            setting([$i => asset('uploads/appLogo/'.$name)])->save();
        }

        return redirect()->back()->with('success', 'App Properties changed successfully');
    }

    /**
     *Change the Application Theme
     */
    public function appThemeUpdate(Request $request)
    {
        $request->validate([
            'app_theme' => 'required',
        ]);
        $data = [
            'app_theme' => $request->app_theme,
            'app_sidebar' => $request->app_sidebar,
            'app_navbar' => $request->app_navbar,
        ];

        $this->updateSettings($data);

        return redirect()->back()->with('success', 'App-theme changed successfully');
    }

    public function appConfigUpdate(Request $request)
    {
        $data = $request->validate([
            'googleanalyticsid' => 'required|string',
            'publisherid' => 'required|string',
            'disqusshortname' => 'required|string',
            'analytics_view_id' => 'required|string',
            'googlemapcode' => 'required|string',
        ]);

        $this->updateSettings($data);

        return redirect()->back()->with('success', 'App Configration changed successfully');
    }

    public function sitemapGenerator()
    {
        $path = public_path('/sitemap.xml');
        SitemapGenerator::create(setting('app_url') ?? env('APP_URL'))->writeToFile($path);

        return redirect()->back()->with('success', 'Sitemap Generate successfully');
    }

    public function changeStatusMaintenance(Request $request)
    {
        if ($request->type == 'maintenance' && user()->id == 1) {
            if ($request->value == 1) {
                Artisan::call('up');

                return back()->with('success', 'Site Live Success');
            } else {
                Artisan::call('down --secret="'.$request->secret.'"');

                return back()->with('success', 'Site Down Success');
            }
        }

        return back()->withErrors(__('app.oops'));
    }

    public function stripePaymentUpdate(Request $request)
    {
        $request->validate([
            'STRIPE_KEY' => 'required|min:10|alpha_dash',
            'STRIPE_SECRET' => 'required|min:10|alpha_dash',
        ]);

        foreach ($request->types as $key => $type) {
            $this->overWriteEnvFile($type, $request[$type]);
        }

        $data = [
            'stripe_key' => $request->STRIPE_KEY,
            'stripe_secret' => $request->STRIPE_SECRET,
            'stripe_status' => ($request->stripe_status == 'on') ? 1 : 0,
        ];

        $this->updateSettings($data);

        return redirect()->back()->with('success', 'Stripe API Keys Updated Successfully');
    }

    public function paypalPaymentUpdate(Request $request)
    {
        $request->validate([
            'PAYPAL_CLIENT_ID' => 'required|min:10|alpha_dash',
            'PAYPAL_CLIENT_SECRET' => 'required|min:10|alpha_dash',
        ]);

        foreach ($request->types as $key => $type) {
            $this->overWriteEnvFile($type, $request[$type]);
        }

        $data = [
            'paypal_client_id' => $request->PAYPAL_CLIENT_ID,
            'paypal_client_secret' => $request->PAYPAL_CLIENT_SECRET,
            'paypal_status' => ($request->paypal_status == 'on') ? 1 : 0,
            'paypal_sandbox' => ($request->paypal_sandbox == 'on') ? 1 : 0,
        ];

        $this->updateSettings($data);

        return redirect()->back()->with('success', 'Paypal API Keys Updated Successfully');
    }

    public function bkashPaymentUpdate(Request $request)
    {
        $request->validate([
            'BKASH_CHECKOUT_APP_KEY' => 'required|string|min:10|alpha_dash',
            'NAGAD_MERCHANT_NUMBER' => 'required|string|min:10|alpha_dash',
            'NAGAD_PG_PUBLIC_KEY' => 'required|string|alpha_dash',
            'NAGAD_MERCHANT_PRIVATE_KEY' => 'required|string|alpha_dash',
        ]);

        foreach ($request->types as $key => $type) {
            $this->overWriteEnvFile($type, $request[$type]);
        }

        // return
        $data = [
            'bkash_app_key' => $request->BKASH_CHECKOUT_APP_KEY,
            'bkash_app_secret' => $request->NAGAD_MERCHANT_NUMBER,
            'bkash_app_username' => $request->NAGAD_PG_PUBLIC_KEY,
            'bkash_app_password' => $request->NAGAD_MERCHANT_PRIVATE_KEY,
            'bkash_status' => ($request->bkash_status == 'on') ? 1 : 0,
            'bkash_sandbox' => ($request->bkash_sandbox == 'on') ? 1 : 0,
        ];

        $this->updateSettings($data);

        return redirect()->back()->with('success', 'Bkash API Keys Updated Successfully');
    }

    public function nagadPaymentUpdate(Request $request)
    {
        $request->validate([
            'NAGAD_MERCHANT_ID' => 'required|string|min:10|alpha_dash',
            'NAGAD_MERCHANT_NUMBER' => 'required|string|min:10|alpha_dash',
            'NAGAD_PG_PUBLIC_KEY' => 'required|string|alpha_dash',
            'NAGAD_MERCHANT_PRIVATE_KEY' => 'required|string|alpha_dash',
        ]);

        foreach ($request->types as $key => $type) {
            $this->overWriteEnvFile($type, $request[$type]);
        }

        // return
        $data = [
            'nagad_merchant_id' => $request->NAGAD_MERCHANT_ID,
            'nagad_merchant_number' => $request->NAGAD_MERCHANT_NUMBER,
            'nagad_public_key' => $request->NAGAD_PG_PUBLIC_KEY,
            'nagad_private_key' => $request->NAGAD_MERCHANT_PRIVATE_KEY,
            'nagad_status' => ($request->nagad_status == 'on') ? 1 : 0,
            'nagad_sandbox' => ($request->nagad_sandbox == 'on') ? 1 : 0,
        ];

        $this->updateSettings($data);

        return redirect()->back()->with('success', 'Nagad API Keys Updated Successfully');
    }

    public function socialLogin(Request $request)
    {
        foreach ($request->types as $key => $type) {
            $this->overWriteEnvFile($type, $request[$type]);
        }
        switch ($request->src) {
            case 'google':
                $data = [
                    'google_login_status' => ($request->google_login_status == 'on') ? 1 : 0,
                    'google_client_id' => $request->GOOGLE_CLIENT_ID,
                    'google_client_secret' => $request->GOOGLE_CLIENT_SECRET,
                    'google_callback' => $request->GOOGLE_CALLBACK,
                ];
                $status = 'Google API Key Updated!';
                break;
            case 'facebook':
                $data = [
                    'facebook_login_status' => ($request->facebook_login_status == 'on') ? 1 : 0,
                    'facebook_client_id' => $request->FACEBOOK_CLIENT_ID,
                    'facebook_client_secret' => $request->FACEBOOK_CLIENT_SECRET,
                    'facebook_callback' => $request->FACEBOOK_CALLBACK,
                ];
                $status = 'Facebook API Key Updated!';
                break;
            case 'github':
                $data = [
                    'github_login_status' => ($request->github_login_status == 'on') ? 1 : 0,
                    'github_client_id' => $request->GITHUB_CLIENT_ID,
                    'github_client_secret' => $request->GITHUB_CLIENT_SECRET,
                    'github_callback' => $request->GITHUB_CALLBACK,
                ];
                $status = 'Github API Key Updated!';
                break;
            case 'nocaptcha':
                $data = [
                    'captcha' => ($request->captcha == 'on') ? 1 : 0,
                    'nocaptcha_sitekey' => $request->NOCAPTCHA_SITEKEY,
                    'nocaptcha_secret' => $request->NOCAPTCHA_SECRET,
                ];
                $status = 'NoCaptcha API Key Updated!';
                break;
            default:
                $data = [
                    'twitter_login_status' => ($request->twitter_login_status == 'on') ? 1 : 0,
                    'twitter_client_id' => $request->TWITTER_CLIENT_ID,
                    'twitter_client_secret' => $request->TWITTER_CLIENT_SECRET,
                    'twitter_callback' => $request->TWITTER_CALLBACK,
                ];
                $status = 'Twitter API Key Updated!';
                break;
        }
        $this->updateSettings($data);

        return redirect()->back()->with('success', $status);
    }

    public function authSettingsUpdate(Request $request)
    {
        $data = [
            '2fa' => ($request->two_factor_auth == 'on') ? 1 : 0,
            'captcha' => ($request->captcha == 'on') ? 1 : 0,
            'email_verification' => ($request->email_verification == 'on') ? 1 : 0,
            'member_register' => ($request->member_register == 'on') ? 1 : 0,
        ];
        $this->updateSettings($data);

        return redirect()->back()->with('success', 'Authentication Settings Updated Successfully');
    }

    private function updateSettings($input)
    {
        foreach ($input as $key => $value) {
            setting([$key => $value])->save();
        }
    }

    /**
     * Run Application Files Backup
     *
     * @param  string  $value
     * @return [type]        [description]
     */
    public function backupFiles()
    {
        Artisan::call('backup:run', ['--only-files' => true]);
        $output = Artisan::output();

        if (Str::contains($output, 'Backup completed!')) {
            return redirect()->back()->with('success', 'Application Files Backed-up successgully');
        } else {
            return redirect()->back()->with('error', 'Application Files Backed-up failed');
        }
    }

    /**
     * Run Application DB Backup
     *
     * @param  string  $value
     * @return [type]        [description]
     */
    public function backupDb()
    {
        Artisan::call('backup:run', ['--only-db' => true]);
        $output = Artisan::output();

        if (Str::contains($output, 'Backup completed!')) {
            return redirect()->back()->with('success', 'Application Database Backed-up successgully');
        } else {
            return redirect()->back()->with('error', 'Application Database Backed-up failed');
        }
    }

    private function getBackups()
    {
        $path = storage_path('app/app-backups');

        // Check if backup-file-path already exist
        if (! File::isDirectory($path)) {
            File::makeDirectory($path, 0777, true, true);
        }

        $files = File::allFiles($path);
        $backups = collect([]);
        foreach ($files as $dt) {
            $backups->push([
                'filename' => pathinfo($dt->getFilename(), PATHINFO_FILENAME),
                'extension' => pathinfo($dt->getFilename(), PATHINFO_EXTENSION),
                'path' => $dt->getPath(),
                'size' => $dt->getSize(),
                'time' => $dt->getMTime(),
            ]);
        }

        return $backups;
    }

    public function downloadBackup($name, $ext)
    {
        $path = storage_path('app/app-backups');
        $file = $path.'/'.$name.'.'.$ext;
        $status = Storage::disk('backup')->download($name.'.'.$ext, $name.'.'.$ext);

        return $status;
    }

    public function deleteBackup($name, $ext)
    {
        $path = storage_path('app/app-backups');
        $file = $path.'/'.$name.'.'.$ext;
        $status = File::delete($file);
        if ($status) {
            return redirect()->back()->with('success', 'Backup deleted successfully');
        } else {
            return redirect()->back()->with('error', 'Ops! an error occured, Try Again');
        }
    }
}
